"""Tests for all sktime detection metrics."""

import pandas as pd
import pytest

from sktime.tests.test_all_estimators import BaseFixtureGenerator, QuickTester


class DetectionMetricFixtureGenerator(BaseFixtureGenerator):
    """Fixture generator for time series detection metric tests.

    Fixtures parameterized
    ----------------------
    estimator_class: estimator inheriting from BaseObject
        ranges over estimator classes not excluded by EXCLUDE_ESTIMATORS, EXCLUDED_TESTS
    estimator_instance: instance of estimator inheriting from BaseObject
        ranges over estimator classes not excluded by EXCLUDE_ESTIMATORS, EXCLUDED_TESTS
        instances are generated by create_test_instance class method
    scenario: instance of TestScenario
        ranges over all scenarios returned by retrieve_scenarios
    """

    # note: this should be separate from TestAllDetectionMetrics
    #   additional fixtures, parameters, etc should be added here
    #   TestAllDetectionMetrics should contain the tests only

    estimator_type_filter = "metric_detection"

    fixture_sequence = [
        "estimator_class",
        "estimator_instance",
        "fitted_estimator",
        "scenario",
    ]


class TestAllDetectionMetrics(DetectionMetricFixtureGenerator, QuickTester):
    """Module level tests for all sktime detection metrics."""

    def test_evaluate_output(self, estimator_instance):
        """Test expected output type of evaluate_output."""
        y_pred = pd.DataFrame({"ilocs": [0, 1, 3, 4, 5]})  # locs are 1, 2, 5, 42, 43
        y_true = pd.DataFrame({"ilocs": [0, 2, 3]})  # locs are 1, 4, 5
        X = pd.DataFrame({"foo": [8, 4, 3, 7, 10, 12]}, index=[1, 2, 4, 5, 42, 43])

        metric = estimator_instance

        requires_X = metric.get_tag("requires_X")
        requires_y_true = metric.get_tag("requires_y_true")

        # test full signature
        loss = metric(y_true, y_pred, X)
        assert isinstance(loss, float)

        # test without X
        if not requires_X:
            loss_no_X = metric(y_true, y_pred)
            assert isinstance(loss_no_X, float)
        else:
            with pytest.raises(TypeError):
                metric(y_true, y_pred)

        # test without y_true
        if not requires_y_true:
            loss_no_y_true_kwarg = metric(y_pred=y_pred)
            assert isinstance(loss_no_y_true_kwarg, float)

            loss_no_y_true_posarg = metric(y_pred)
            assert isinstance(loss_no_y_true_posarg, float)

            assert loss_no_y_true_kwarg == loss_no_y_true_posarg
        else:
            with pytest.raises(TypeError):
                metric(y_pred)
            with pytest.raises(TypeError):
                metric(y_pred=y_pred)
