/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 *
 *   SPDX-License-Identifier: Apache-2.0
 */

package org.apache.jena.sparql.exec;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.function.Consumer;

import org.apache.jena.atlas.iterator.Iter;
import org.apache.jena.sparql.core.Var;
import org.apache.jena.sparql.engine.binding.Binding;

public class RowSetStream implements RowSet {

    private Iterator<Binding> iterator ;
    private List<Var> resultVars ;
    private int rowNumber ;

    public static RowSet create(List<Var> vars, Iterator<Binding> iterator) {
        return new RowSetStream(vars, iterator);
    }

    protected RowSetStream(List<Var> resultVars, Iterator<Binding> bindings) {
        this.iterator = bindings;
        this.resultVars = new ArrayList<>(resultVars);
        rowNumber = 0;
    }

    @Override
    public boolean hasNext() {
        return iterator.hasNext();
    }

    @Override
    public Binding next() {
        rowNumber++;
        return iterator.next();
    }

    @Override
    public List<Var> getResultVars() {
        return resultVars;
    }

    @Override
    public RowSetRewindable rewindable() {
        return RowSetMem.create(this);
    }

    @Override
    public long getRowNumber() {
        return rowNumber;
    }

    @Override
    public void close() {
        Iter.close(iterator);
    }

    @Override
    public void forEachRemaining(Consumer<? super Binding> action) {
        iterator.forEachRemaining(b -> {
            ++rowNumber;
            action.accept(b);
        });
    }
}
