/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.openjpa.persistence.jdbc.unique;

import java.util.Collection;

import jakarta.persistence.Column;
import jakarta.persistence.Entity;
import jakarta.persistence.Id;
import jakarta.persistence.JoinColumn;
import jakarta.persistence.JoinTable;
import jakarta.persistence.ManyToMany;
import jakarta.persistence.SecondaryTable;
import jakarta.persistence.Table;
import jakarta.persistence.UniqueConstraint;

/**
 * Data structures for testing unique constraint settings on ORM Annotations.
 * @UniqueConstraint annotation is declared at class-level with
 *    @Table,
 *    @SecondaryTable annotations
 * and at field-level with
 *    @JoinTable annotation.
 * also with
 *    @Column(unique=true) on single column.
 *
 * The columns included in unique constraint must be non-nullable. This is
 * recommended that the non-nullability of the column is explictly set by the
 * user, though the implementation forces a column to non-nullable as a column
 * is included in a unique constraint.
 *
 * The name of the constraint is generated by the implementation as JPA ORM
 * specification has not allowed to specify a name for the constraint via the
 * annotation or XML descriptor. Some databases allow two constraints having the
 * same name but applied to different tables, while some other databases do not.
 *
 *
 * @author Pinaki Poddar
 *
 */
@Entity
@Table(name="N_UNIQUE_A",
	   uniqueConstraints={@UniqueConstraint(name="uca_f1_f2", columnNames={"f1","f2"}),
		                  @UniqueConstraint(name="uca_f3_f4", columnNames={"f3","f4"})})
@SecondaryTable(name="N_UNIQUE_SECONDARY",
		uniqueConstraints=@UniqueConstraint(name="uca_sf1", columnNames={"sf1"}))

public class NamedUniqueA {
	@Id
	private int aid;

	// Same named field in UniqueB also is defined as unique
	@Column(unique=true, nullable=false)
	private int f1;

	@Column(nullable=false)
	private int f2;

	@Column(nullable=false)
	private int f3;

	@Column(nullable=false)
	private int f4;

	private int f5;
	private int f6;

	@Column(table="N_UNIQUE_SECONDARY", nullable=false)
	private short sf1;

	@Column(table="N_UNIQUE_SECONDARY")
	private short sf2;

	@ManyToMany
	@JoinTable(name="N_UNIQUE_JOINTABLE",
			joinColumns={@JoinColumn(name="FK_A", nullable=false,
					referencedColumnName="aid")},
            inverseJoinColumns={@JoinColumn(name="FK_B", nullable=false,
					referencedColumnName="bid")},
            uniqueConstraints=@UniqueConstraint(name="uca_fka_fkb", columnNames={"FK_A","FK_B"}))
	private Collection<NamedUniqueB> bs;
}
